'
' SAP Rework Status and Tool Type Validation Script
' Author: Dario Pascoal
'
' Description: This VBScript automates the process of checking material rework 
' requirements and tool type classification within the SAP ZMEC transaction system. 
' The script serves as a critical validation point in the DHL tool management 
' workflow, determining both whether tools require rework processing and their 
' classification type for proper handling procedures.
'
' Business Purpose:
' In DHL's tool management process, different tools have different processing
' requirements based on their rework status and type classification. This script
' automates the dual validation by reading specific status and type fields in SAP
' that determine processing workflows. Tools marked with "U3" status require rework
' procedures, while tool types (Z007=Tool, Z004=Consumable) determine handling protocols.
'
' Workflow Integration:
' 1. Receives material number from calling application
' 2. Connects to active SAP GUI session
' 3. Navigates to ZMEC transaction for material analysis
' 4. Searches for specified material number with plant/org context
' 5. Extracts rework status and tool type classification data
' 6. Analyzes status against rework requirements (U3 = needs rework)
' 7. Classifies tool type for appropriate handling procedures
' 8. Returns dual status codes for workflow routing decisions
'
' Technical Implementation:
' - VBScript with COM object automation for SAP GUI interaction
' - Dual field extraction for status and type classification
' - Command-line interface with material number and logging parameters
' - Comprehensive error handling with graceful degradation
' - File-based logging system for audit trails and debugging
' - Registry security configuration for unattended automation
'
' Integration Points:
' - SAP GUI Scripting Engine for automated navigation and data extraction
' - DHL All-in-One main application for status reporting and workflow control
' - Windows Shell for security dialog handling and registry management
' - File System Object for logging operations and error tracking
' - ZMEC transaction for material rework status and type classification
'
' Security Features:
' - Temporary registry modifications for automation enablement
' - Error handling to prevent script crashes and data corruption
' - Detailed logging for security audit trails and compliance
' - Secure SAP GUI connection handling with proper authentication
' - Automatic cleanup procedures to prevent resource leaks
'
' Return Codes:
' Rework Status:
' - "needs-rework": Material has U3 status requiring rework procedures
' - "no-rework": Material does not require rework processing
' Tool Type Classification:
' - "TOOL_TYPE:tool": Material is classified as Z007 (Tool)
' - "TOOL_TYPE:consumable": Material is classified as Z004 (Consumable)
' - "TOOL_TYPE:unknown-type": Material has unrecognized or missing type
'

' =============================================================================
' GLOBAL ERROR HANDLING CONFIGURATION
' =============================================================================
' Enable comprehensive error handling to prevent script crashes and provide
' graceful degradation when encountering unexpected SAP GUI conditions
On Error Resume Next

' =============================================================================
' FILE SYSTEM INITIALIZATION
' =============================================================================
' Initialize file system objects for logging and temporary file operations
' This section establishes the foundation for all file-based operations

' Initialize FileSystemObject for file and logging operations
' This object handles all file system interactions throughout the script
Dim fso, logFile, logPath
Set fso = CreateObject("Scripting.FileSystemObject")

' Handle FileSystemObject creation errors
' If this fails, the script cannot proceed as logging is essential
If Err.Number <> 0 Then
    WScript.Echo "ERROR: Failed to create FileSystemObject - " & Err.Description
    WScript.Quit 1
End If

' =============================================================================
' LOGGING SYSTEM SETUP
' =============================================================================
' Establish comprehensive logging system for operation tracking and debugging
' The logging system supports both temporary and specified log file locations

' Create default log path in system temporary directory
' This ensures logging is available even if no specific path is provided
logPath = fso.GetSpecialFolder(2) & "\sap_rework_and_tool_type_check.log"
WScript.Echo "DEBUG: Log path will be: " & logPath

' Override log path if provided as command line argument
' This allows external applications to specify custom log locations
If WScript.Arguments.Count > 1 Then
    logPath = WScript.Arguments.Item(1)
    
    ' Attempt to open specified log file in append mode
    ' Append mode preserves existing log entries for continuous tracking
    On Error Resume Next
    Set logFile = fso.OpenTextFile(logPath, 8, True)
    
    ' Handle log file opening errors gracefully
    ' Continue execution even if custom logging fails
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Log script startup information for audit trail
        ' This creates clear session markers in the log file
        LogMessage "========================================="
        LogMessage "Rework and Tool Type Check Script Started at " & Now
        LogMessage "Script: " & WScript.ScriptName
        LogMessage "Material Number: " & WScript.Arguments.Item(0)
        LogMessage "========================================="
    End If
End If

' =============================================================================
' FALLBACK LOG FILE CREATION
' =============================================================================
' Ensure log file availability through fallback creation mechanisms
' This guarantees logging functionality regardless of initial setup success

' Attempt to open or create the log file with comprehensive error handling
' This section ensures logging is always available for the script execution
On Error Resume Next
Set logFile = fso.OpenTextFile(logPath, 8, True)

' Handle log file access failures with fallback creation
' If opening fails, attempt to create a completely new log file
If Err.Number <> 0 Then
    WScript.Echo "ERROR: Failed to open log file - " & Err.Description
    WScript.Echo "Attempting to create new log file..."
    
    ' Attempt to create a new log file with overwrite permission
    ' This is the final fallback to ensure logging functionality
    On Error Resume Next
    Set logFile = fso.CreateTextFile(logPath, True)
    
    ' If log file creation also fails, terminate with error
    ' Logging is considered essential for audit and debugging purposes
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Failed to create log file - " & Err.Description
        WScript.Quit 1
    End If
End If

' =============================================================================
' LOGGING UTILITY FUNCTIONS
' =============================================================================
' Centralized logging functions to ensure consistent operation tracking
' and error reporting across all script operations and SAP interactions

'
' Primary logging function for dual-output message recording
' This function writes messages to both log file and console for
' comprehensive tracking and immediate user feedback
'
' Parameters:
' - message: String containing the message to be logged
'
' Behavior:
' - Writes timestamped entry to log file if available
' - Displays message on console for immediate feedback
' - Handles logging errors gracefully without stopping execution
'
Sub LogMessage(message)
    On Error Resume Next
    ' Write to log file with timestamp and script identification
    ' Format: timestamp - [script name] - message for easy parsing
    If Not logFile Is Nothing Then
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
        ' Handle log writing errors without stopping execution
        If Err.Number <> 0 Then
            WScript.Echo "ERROR: Failed to write to log - " & Err.Description
        End If
    End If
    ' Display message to console for immediate user feedback
    ' This ensures visibility even if file logging fails
    WScript.Echo message
End Sub

'
' Cleanup function to properly close log file and release system resources
' This function ensures proper log file closure and resource cleanup to
' prevent memory leaks and ensure all log data is properly written to disk
'
Sub CloseLogFile()
    On Error Resume Next
    ' Close log file if it's currently open and available
    If Not logFile Is Nothing Then
        logFile.Close
        Set logFile = Nothing
    End If
    ' Reset error handling to default state
    On Error Goto 0
End Sub

' =============================================================================
' SCRIPT INITIALIZATION AND VALIDATION
' =============================================================================
' Initialize script execution with logging validation and material number setup
' This section establishes the operational context for the entire workflow

' Test logging functionality and log script startup
' This validates that logging is working correctly before proceeding
LogMessage "=== Rework and Tool Type Check Started ==="
LogMessage "Log file initialized at: " & logPath

' Extract and validate material number from command line arguments
' The material number is the primary input for all SAP operations
LogMessage "Checking material number: " & WScript.Arguments.Item(0)

' =============================================================================
' ENVIRONMENT INITIALIZATION
' =============================================================================
' Initialize Windows Script Host shell object for system interactions
' This object handles registry modifications and security dialog automation

' Create Windows Script Host shell object for registry and system operations
' This enables registry modifications needed for SAP GUI automation
Set WSHShell = CreateObject("WScript.Shell")

' =============================================================================
' SAP GUI SECURITY CONFIGURATION
' =============================================================================
' Configure SAP GUI security settings to enable unattended automation
' These registry modifications prevent security dialogs that would halt execution

' Disable SAP GUI scripting security warnings in registry
' These settings are essential for automated script execution
On Error Resume Next

' Disable attachment warnings when scripts connect to SAP GUI sessions
' This prevents popup dialogs that would halt automation processes
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"

' Disable connection warnings for script-to-SAP communication
' This ensures seamless connection establishment without user intervention
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"

' Handle registry modification errors gracefully
' Log errors but continue execution as these settings may already be configured
If Err.Number <> 0 Then
    LogMessage "Error setting registry keys: " & Err.Description
    Err.Clear
End If

' =============================================================================
' COMMAND LINE ARGUMENT PROCESSING
' =============================================================================
' Extract and validate command line arguments for material number processing
' The material number is essential for all subsequent SAP operations

' Get material number from first command line argument (required)
' This material number identifies the equipment for rework and type validation
materialNumber = WScript.Arguments.Item(0)
LogMessage "Material number received: " & materialNumber

' =============================================================================
' SAP GUI CONNECTION ESTABLISHMENT
' =============================================================================
' Establish connection to SAP GUI application and active session
' This section handles the complex process of connecting to SAP through
' the SAP GUI Scripting Engine with comprehensive error handling

' Initialize SAP GUI connection with error handling
' This establishes the foundation for all SAP automation operations
On Error Resume Next

' Initialize SAP GUI application object if not already connected
' The application object is the top-level SAP GUI automation interface
If Not IsObject(application) Then
   ' Connect to the SAP GUI Automation Server
   ' This server manages all SAP GUI automation and scripting operations
   Set SapGuiAuto = GetObject("SAPGUI")
   
   ' Handle SAP GUI connection errors
   ' If SAP is not running or accessible, terminate with error
   If Err.Number <> 0 Then
       LogMessage "Error getting SAPGUI: " & Err.Description
       WScript.Echo "ERROR: Cannot connect to SAP - " & Err.Description
       WScript.Quit 1
   End If
   
   ' Get the scripting engine from the SAP GUI server
   ' The scripting engine provides programmatic access to SAP GUI elements
   Set application = SapGuiAuto.GetScriptingEngine
   
   ' Handle scripting engine initialization errors
   ' If scripting is disabled or unavailable, terminate with error
   If Err.Number <> 0 Then
       LogMessage "Error getting scripting engine: " & Err.Description
       WScript.Echo "ERROR: Cannot initialize SAP scripting - " & Err.Description
       WScript.Quit 1
   End If
End If

LogMessage "SAP GUI initialized successfully"

' Initialize SAP connection object if not already established
' The connection object represents a specific SAP system connection
If Not IsObject(connection) Then
   ' Get the first (primary) connection from available connections
   ' Children(0) accesses the first active connection to SAP
   Set connection = application.Children(0)
   
   ' Handle connection access errors
   ' If no active connections exist, terminate with error
   If Err.Number <> 0 Then
       LogMessage "Error getting connection: " & Err.Description
       WScript.Echo "ERROR: No SAP connection available - " & Err.Description
       WScript.Quit 1
   End If
End If

' Initialize SAP session object for user interaction simulation
' The session object provides access to SAP screens and transactions
If Not IsObject(session) Then
   ' Get the first (primary) session from the active connection
   ' This session will be used for all SAP GUI automation commands
   Set session = connection.Children(0)
   
   ' Handle session access errors
   ' If no active sessions exist, terminate with error
   If Err.Number <> 0 Then
       LogMessage "Error getting session: " & Err.Description
       WScript.Echo "ERROR: No SAP session available - " & Err.Description
       WScript.Quit 1
   End If
End If

' =============================================================================
' SAP AUTOMATION WORKFLOW EXECUTION
' =============================================================================
' Execute the complete rework status and tool type validation workflow
' This section orchestrates all SAP GUI interactions to extract material data

' Begin SAP command execution with comprehensive error handling
' All SAP interactions are logged for audit and debugging purposes
On Error Resume Next
LogMessage "Executing SAP commands..."

' =============================================================================
' SAP TRANSACTION NAVIGATION
' =============================================================================
' Navigate to ZMEC transaction for material rework and type analysis
' ZMEC provides access to material-specific status and classification data

' Maximize SAP GUI window to ensure all elements are visible
' This prevents issues with hidden or partially visible UI elements
session.findById("wnd[0]").maximize
LogMessage "Window maximized"

' Navigate to ZMEC transaction using the command field
' The transaction code directs SAP to the material evaluation interface
session.findById("wnd[0]/tbar[0]/okcd").text = "zmec"
LogMessage "Entered transaction code: zmec"

' Execute the transaction command (equivalent to pressing Enter)
' This loads the ZMEC transaction interface for material analysis
session.findById("wnd[0]").sendVKey 0
LogMessage "Executed transaction"

' =============================================================================
' MATERIAL SEARCH PARAMETER SETUP
' =============================================================================
' Configure search parameters for material rework and type lookup
' This section sets up the context for material-specific data extraction

' Enter the material number in the material search field
' P_MATNR is the primary material number field in ZMEC transaction
session.findById("wnd[0]/usr/ctxtP_MATNR").text = materialNumber
LogMessage "Entered material number: " & materialNumber

' Set plant code to NL01 for Netherlands operations context
' P_WERKS specifies the plant where the material is managed
session.findById("wnd[0]/usr/ctxtP_WERKS").text = "nl01"

' Set sales organization to NL01 for consistent organizational context
' P_VKORG specifies the sales organization for material classification
session.findById("wnd[0]/usr/ctxtP_VKORG").text = "nl01"
LogMessage "Set plant and sales org to nl01"

' Execute the search using function key F8 (Execute)
' This triggers the material lookup and displays status/type information
session.findById("wnd[0]").sendVKey 8
LogMessage "Executed search"

' =============================================================================
' REWORK STATUS DATA EXTRACTION
' =============================================================================
' Extract rework status information from the ZMEC transaction results
' This section handles the critical rework status determination logic

' Focus on the rework status field to ensure reliable data extraction
' lbl[138,5] contains the rework status code that determines processing workflow
session.findById("wnd[0]/usr/lbl[138,5]").setFocus

' Handle rework status field access errors
' If the field cannot be accessed, terminate with error as this is critical data
If Err.Number <> 0 Then
    LogMessage "Error accessing rework status field: " & Err.Description
    WScript.Echo "ERROR: Cannot access rework status - " & Err.Description
    WScript.Quit 1
End If

' Extract the rework status from the focused field position
' This status code determines whether the material requires rework processing
reworkStatus = session.findById("wnd[0]/usr/lbl[138,5]").text
LogMessage "Retrieved rework status: " & reworkStatus

' =============================================================================
' TOOL TYPE DATA EXTRACTION
' =============================================================================
' Extract tool type classification from the same ZMEC transaction screen
' This section handles the tool type determination for workflow routing

' Extract tool type information from the classification field
' lbl[131,5] contains the tool type code for material classification
On Error Resume Next
toolType = session.findById("wnd[0]/usr/lbl[131,5]").text

' Handle tool type field access errors gracefully
' If tool type cannot be determined, use "UNKNOWN" for safe processing
If Err.Number <> 0 Then
    LogMessage "Error accessing tool type field: " & Err.Description
    toolType = "UNKNOWN"
Else
    LogMessage "Retrieved tool type: " & toolType
End If
On Error Resume Next

' =============================================================================
' REWORK STATUS ANALYSIS AND ROUTING
' =============================================================================
' Analyze rework status and make workflow routing decisions based on
' business rules for material rework processing requirements

' Check if rework status indicates rework requirement (U3 status)
' U3 is the specific status code that indicates rework is needed
If Trim(reworkStatus) = "U3" Then
    LogMessage "Status is U3 - needs rework"
    ' Return rework required status to calling application
    ' This will route the material through rework processing workflow
    WScript.Echo "needs-rework"
Else
    LogMessage "Status is not U3 - no rework needed"
    ' Return no rework needed status to calling application
    ' This allows the material to proceed with standard processing
    WScript.Echo "no-rework"
End If

' =============================================================================
' TOOL TYPE CLASSIFICATION AND ROUTING
' =============================================================================
' Analyze tool type classification and provide routing information based on
' material type codes that determine handling procedures and workflows

' Log tool type analysis results for audit trail
LogMessage "Tool type check result: " & Trim(toolType)

' Analyze tool type and provide classification routing
' Different tool types require different handling procedures in DHL workflow
If Trim(toolType) = "Z007" Then
    ' Z007 indicates standard tool classification
    LogMessage "Tool type is Z007 - Tool"
    ' Return tool type classification for workflow routing
    WScript.Echo "TOOL_TYPE:tool"
ElseIf Trim(toolType) = "Z004" Then
    ' Z004 indicates consumable material classification
    LogMessage "Tool type is Z004 - Consumable"
    ' Return consumable type classification for different handling procedures
    WScript.Echo "TOOL_TYPE:consumable"
Else
    ' Unknown or unrecognized tool type requires special handling
    LogMessage "Tool type is unknown or other: " & toolType
    ' Return unknown type status for manual review or default processing
    WScript.Echo "TOOL_TYPE:unknown-type"
End If

' =============================================================================
' SAP TRANSACTION CLEANUP
' =============================================================================
' Perform proper SAP transaction cleanup to ensure clean session state
' This section ensures SAP returns to a proper state for subsequent operations

' Close the current transaction using standard SAP navigation
' These function keys perform the equivalent of menu-based exit operations
session.findById("wnd[0]").sendVKey 2      ' F2: Exit current screen
session.findById("wnd[0]").sendVKey 12     ' F12: Cancel/Exit transaction
session.findById("wnd[0]").sendVKey 12     ' F12: Additional exit for transaction cleanup
LogMessage "Transaction closed"

' =============================================================================
' SCRIPT FINALIZATION AND CLEANUP
' =============================================================================
' Complete script execution with proper logging closure and resource cleanup
' This section ensures all resources are properly released

' Log successful completion of rework and tool type validation
LogMessage "=== Rework and Tool Type Check Completed ==="

' Execute final cleanup procedures to close log file and release resources
' This ensures proper resource management and log data persistence
CloseLogFile()
